# Copyright (C) 2016, 2017, 2018, 2019, 2020 The Meme Factory, Inc.
# http://www.karlpinc.com/

# This file is part of PGWUI.
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation, either version 3 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#

# Karl O. Pinc <kop@karlpinc.com>

#
# Common Makefile for PGWUI projects
#

# Variables to customize:
# PGWUI_COMPONENT := pgwui_example


# This works on debian...
PYTHON_EXE=python3
VIRTUALENV=virtualenv -p ${PYTHON_EXE}
# (Optional) where pyenv is installed (git clone somepath)
# For running tox against multiple python versions
PYENV_INSTALLATION := $(HOME)/.pyenv
PYENV_BIN := $(PYENV_INSTALLATION)/bin

TARGETS =
EXTRA_TARGETS = README.html
SETUPTOOLS_STUFF = build dist src/${PGWUI_COMPONENT}.egg-info
PYTEST_STUFF = .cache                                      \
            src/${PGWUI_COMPONENT}/__pycache__            \
            src/${PGWUI_COMPONENT}/views/__pycache__      \
            tests/__pycache__ tests/views/__pycache__
COVERAGE_STUFF = .coverage
TOX_STUFF = .tox

include help.mk

all: ${TARGETS}

##
## Usage: make TARGET
##
## TARGET is one of:
#

## README.html         Make html version of README.rst
README.html: README.rst
	rst2html --strict README.rst > README.html

## publish             Update all public-facing info
##                     ("check-manifest" + "upload" + "push")
.PHONY: publish
publish: check-manifest upload push

## run_tests           Run regression tests
.PHONY: run_tests
run_tests: devel/testenv
	if [ -x $(PYENV_BIN)/pyenv ] ; then \
	  (set -e ; \
	   export PYENV_ROOT=$(PYENV_INSTALLATION) ; \
	   export PATH="$(PYENV_BIN):$$PATH" ; \
	   bash -c 'set -e ; \
	            eval "$$(pyenv init -)" ; \
	            pyenv shell system $$(pyenv versions --bare) ; \
	            devel/testenv/bin/tox --skip-missing-interpreters' ) ; \
	else \
	  devel/testenv/bin/tox --skip-missing-interpreters ; \
	fi

## dist                Create sdist Python package in ./dist
.PHONY: dist
dist:
	rm -rf dist
	${PYTHON_EXE} setup.py sdist


# Expected targets

## clean               Removed generated data
.PHONY: clean
clean:
	#$(MAKE) -C docs clean
	rm -rf ${TARGETS} ${EXTRA_TARGETS} ${SETUPTOOLS_STUFF} \
	       ${PYTEST_STUFF} ${COVERAGE_STUFF} ${TOX_STUFF} devel

##
## Less used TARGETS are:
#


# Useless targets

## register_test       Register project at pypi test site
.PHONY: register_test
register_test:
	${PYTHON_EXE} setup.py register -r https://testpypi.python.org/pypi

## register            Register project at pypi live site
.PHONY: register
register:
	${PYTHON_EXE} setup.py register


# Not so useless targets, used by the package owners for package management

## upload_test         Upload project to pypi test site
.PHONY: upload_test
upload_test:
	${PYTHON_EXE} setup.py sdist upload -r https://testpypi.python.org/pypi

## upload              Upload project to pypi live site
.PHONY: upload
upload:
	${PYTHON_EXE} setup.py sdist upload

## push                Push repo changes to public repo
##                     (Hooks at bitbucket update readthedocs)
.PHONY: push
	git push

## check               Quick run of all regression tests,
##                     only with the default python
.PHONY: check
check: devel/pytest
	devel/pytest/bin/flake8 .
	devel/pytest/bin/py.test --cov=${PGWUI_COMPONENT} tests

## check-unit          Quick run of unit regression tests,
##                     only with the default python
.PHONY: check-unit
check-unit: devel/pytest
	devel/pytest/bin/flake8 .
	devel/pytest/bin/py.test -m unittest --cov=${PGWUI_COMPONENT} tests

## check-integration   Quick run of integration regression tests,
##                     only with the default python
.PHONY: check-integration
check-integration: devel/pytest
	devel/pytest/bin/flake8 .
	devel/pytest/bin/py.test -m integrationtest \
	                         --cov=${PGWUI_COMPONENT} tests

## check-debug         Start a regression test and drop into the Pdb
##                     debugger.  Variables designate the test file
##                     and test function, i.e.:
##                        FILE=test_mymodule.py TEST=test_somefunction \
##                          make check-debug
.PHONY: check-debug
check-debug: devel/pytest
	devel/pytest/bin/flake8 .
	devel/pytest/bin/py.test --trace tests/${FILE}::${TEST}

## pudb                Run the python pudb debugger
.PHONY: pudb
pudb: devel/pudb
	# echo 'import pudb; pu.db' | devel/pudb/bin/python
	devel/pudb/bin/python

## devel/check-manifest
##                     Create a check-manifest venv
devel/check-manifest: devel
	[ -d devel/check-manifest ] \
	  || ( ${VIRTUALENV} devel/check-manifest ; \
	       devel/check-manifest/bin/pip install --upgrade pip ; \
	       devel/check-manifest/bin/pip install --upgrade setuptools ; \
	       devel/check-manifest/bin/pip install check-manifest )

## check-manifest      Run check-manifest
check-manifest: devel/check-manifest
	devel/check-manifest/bin/check-manifest

## update_check-manifest
##                     Upgrade check-manifest venv to latest
##                     versions
update_check-manifest: devel/check-manifest
	devel/check-manifest/bin/pip -U

## update_testenv      Upgrade test venv to latest versions
update_testenv: devel/testenv
	devel/testenv/bin/pip -U

## help                Show this help
##
## This Makefile does not work with the docs.  See:
## cd docs; make help


# Development related targets

# Re-create development environment when setup.py changes
devel: setup.py
	rm -rf devel ${TOX_STUFF}
	mkdir devel

# virtualenv for development
devel/testenv: devel
	[ -d devel/testenv ] \
	  || ( ${VIRTUALENV} devel/testenv ; \
	       devel/testenv/bin/pip install --upgrade pip ; \
	       devel/testenv/bin/pip install --upgrade setuptools ; \
	       devel/testenv/bin/pip install tox ; \
	       devel/testenv/bin/pip install -e '.[testing]' ; \
	     )

# virtualenv for pytest
devel/pytest: devel dist
	if [ ! -d devel/pytest ] ; then \
	   ( ${VIRTUALENV} devel/pytest ; \
	     devel/pytest/bin/pip install --upgrade pip ; \
	     devel/pytest/bin/pip install --upgrade setuptools ; \
	     devel/pytest/bin/pip install --upgrade wheel ; \
	     devel/pytest/bin/pip install dist/${PGWUI_COMPONENT}-*.tar.gz ; \
	     devel/pytest/bin/pip install flake8 ; \
	     devel/pytest/bin/pip install pytest-cov ; \
	     devel/pytest/bin/pip install -e '.[testing]' ; \
	   ) \
	else \
	   ( devel/pytest/bin/pip uninstall -y ${PGWUI_COMPONENT} ; \
	     devel/pytest/bin/pip install dist/${PGWUI_COMPONENT}-*.tar.gz ; \
	     devel/pytest/bin/pip install -e '.[testing]' ; \
	     ) ; \
	fi

# virtualenv for pudb
devel/pudb: devel dist
	if [ ! -d devel/pudb ] ; then \
	  ( ${VIRTUALENV} devel/pudb ; \
	    devel/pudb/bin/pip install --upgrade pip ; \
	    devel/pudb/bin/pip install --upgrade setuptools ; \
	    devel/pudb/bin/pip install --upgrade wheel ; \
	    devel/pudb/bin/pip install dist/${PGWUI_COMPONENT}-*.tar.gz ; \
	    devel/pudb/bin/pip install -e '.[testing]' ; \
	    devel/pudb/bin/pip install pudb ; \
	  ) \
	else \
	  ( devel/pudb/bin/pip uninstall -y ${PGWUI_COMPONENT} ; \
	    devel/pudb/bin/pip install -e '.[testing]' ; \
	    devel/pudb/bin/pip install dist/${PGWUI_COMPONENT}-*.tar.gz \
	  ) ; \
	fi
