# Copyright (C) 2018, 2020, 2021 The Meme Factory, Inc.
# http://www.karlpinc.com/

# This file is part of PGWUI_Common.
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation, either version 3 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#

# Karl O. Pinc <kop@karlpinc.com>

import pytest
from pyramid.threadlocal import get_current_request

import pgwui_common.view as view

from pgwui_develop import testing

# Activiate the PGWUI pytest plugin
pytest_plugins = ("pgwui",)

# Mark all tests with "unittest"
pytestmark = pytest.mark.unittest


# Helper functions and constants

FOO_URL = 'foo://bar/'

mock_find_pgwui_components = testing.make_mock_fixture(
    view, 'find_pgwui_components')


def mock_view(request):
    if (hasattr(request, 'registry')
            and 'pgwui' in request.registry.settings):
        return request.registry.settings
    return {'pgwui': {'foo': FOO_URL}}


def check_base_view_results(request, pgwui):
    assert pgwui['foo'] == FOO_URL


# Unit tests

# merge_urls()

def test_merge_urls(pyramid_request_config):
    '''The urls in the pgwui response dict is updated with
    the urls in the request's settings
    '''
    old_urls = {'key1': 'val1'}
    new_urls = {'key2': 'val2', 'key3': 'val3'}
    request = get_current_request()
    request.registry.settings = {'pgwui': {'urls': new_urls}}
    pgwui = {'urls': old_urls.copy()}
    view.merge_urls(request, pgwui)

    expected_urls = old_urls.copy()
    expected_urls.update(new_urls)
    assert pgwui['urls'] == expected_urls


mock_merge_urls = testing.make_mock_fixture(
    view, 'merge_urls')


# base_view()
def test_base_view_urls(mock_merge_urls):
    '''The response has the 'pgwui['urls']' dict added to it by merge_urls()
    '''
    expected_urls = {'key1': 'val1', 'key2': 'val2'}

    def mock_view(request):
        return {}

    def fake_merge_urls(request, pgwui):
        pgwui['urls'] = expected_urls

    mock_merge_urls.side_effect = fake_merge_urls
    wrapper = view.base_view(mock_view)
    response = wrapper(get_current_request())

    assert 'pgwui' in response
    pgwui = response['pgwui']
    assert 'urls' in pgwui
    assert pgwui['urls'] == expected_urls


def test_base_view_default(mock_merge_urls):
    '''The response retains the mock view's variables'''
    wrapper = view.base_view(mock_view)
    request = get_current_request()
    response = wrapper(request)
    pgwui = response['pgwui']
    check_base_view_results(request, pgwui)


mock_base_view = testing.make_mock_fixture(view, 'base_view')


# errors_base_view()

def test_errors_base_view(mock_base_view):
    '''Wrapper calls base_view()
    '''
    wrapper = view.errors_base_view(mock_view)
    request = get_current_request()
    wrapper(request)

    mock_base_view.assert_called_once()


mock_errors_base_view = testing.make_mock_fixture(view, 'errors_base_view')


# auth_base_view()

def test_auth_base_view(mock_errors_base_view):
    '''Wrapper calls errors_base_view()
    '''
    wrapper = view.auth_base_view(mock_view)
    request = get_current_request()
    wrapper(request)

    mock_errors_base_view.assert_called_once()
