# Copyright (C) 2020, 2021 The Meme Factory, Inc.  http://www.karlpinc.com/

# This file is part of PGWUI_Common.
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation, either version 3 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#

# Karl O. Pinc <kop@karlpinc.com>

import pytest
from pyramid.threadlocal import get_current_request

import pgwui_common.exceptions as common_ex
import pgwui_common.views.page_views as page_views

from pgwui_develop import testing

# Activiate our pytest plugin
# pytest_plugins = ("pgwui",)


# Helper functions and constants

mock_page = testing.instance_method_mock_fixture('page')
mock_FileResponse = testing.make_mock_fixture(
    page_views, 'FileResponse')


# Unit tests

# PageViewer.home_page()

@pytest.mark.unittest
def test_home_page(pyramid_request_config, mock_page):
    '''Called with correct name
    '''
    request = get_current_request()
    view = page_views.PageViewer(request)
    mocked_page = mock_page(view)
    view.home_page()
    assert mocked_page.call_args[0][0] == 'home_page'


# PageViewer.menu_page()

@pytest.mark.unittest
def test_menu_page(pyramid_request_config, mock_page):
    '''Called with correct name
    '''
    request = get_current_request()
    view = page_views.PageViewer(request)
    mocked_page = mock_page(view)
    view.menu_page()
    assert mocked_page.call_args[0][0] == 'menu_page'


# PageViewer.page()

@pytest.mark.unittest
def test_page_success(pyramid_request_config, mock_FileResponse):
    '''FileResponse() called
    '''
    expected = 'some value'
    mock_FileResponse.return_value = expected

    pgwui = {'test_page': {'source': 'anything'}}
    request = get_current_request()
    request.registry.settings['pgwui'] = pgwui
    view = page_views.PageViewer(request)
    result = view.page('test_page')

    mock_FileResponse.assert_called_once()
    assert result == expected


@pytest.mark.parametrize(
    ('ex', 'expected'), [
        (FileNotFoundError, common_ex.BadPageFileNotFoundError),
        (PermissionError, common_ex.BadPageFilePermissionError),
        (IsADirectoryError, common_ex.BadPageIsADirectoryError)])
@pytest.mark.unittest
def test_page_exception(pyramid_request_config, mock_FileResponse,
                        ex, expected):
    '''The correct exception is raised
    '''
    mock_FileResponse.side_effect = ex

    pgwui = {'test_page': {'source': 'anything'}}
    request = get_current_request()
    request.registry.settings['pgwui'] = pgwui
    view = page_views.PageViewer(request)
    with pytest.raises(expected):
        view.page('test_page')

    mock_FileResponse.assert_called_once()
