# Copyright (C) 2018, 2019, 2020, 2021 The Meme Factory, Inc.
# http://www.karlpinc.com/

# This file is part of PGWUI_Server.
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation, either version 3 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#

# Karl O. Pinc <kop@karlpinc.com>

import pytest

import pgwui_common.exceptions as ex
import pgwui_common.checkset as checkset


# require_settings()

@pytest.mark.unittest
def test_require_settings_good():
    '''No errors when the required settings are in the config
    '''
    required = ['settinga', 'settingb']
    settings = {'settinga': 'a', 'settingb': 'b'}

    result = checkset.require_settings('testcomp', required, settings)

    assert result == []


@pytest.mark.unittest
def test_require_settings_bad():
    '''Errors when the required settings are not in the config
    '''
    required = ['settinga', 'settingb']
    settings = {}

    result = checkset.require_settings('testcomp', required, settings)

    assert len(result) == len(required)
    for error in result:
        assert isinstance(error, ex.MissingSettingError)


# unknown_settings()

@pytest.mark.unittest
def test_unknown_settings_good():
    '''There are no errors when all settings are known
    '''
    settings = ['settinga', 'settingb']
    conf = {'settinga': 'a', 'settingb': 'b'}

    result = checkset.unknown_settings('testcomp', settings, conf)

    assert result == []


@pytest.mark.unittest
def test_unknown_settings_bad():
    '''Errors when settings are not known
    '''
    conf = {'settinga': 'a', 'settingb': 'b'}

    result = checkset.unknown_settings('testcomp', [], conf)

    assert len(result) == len(conf)
    for error in result:
        assert isinstance(error, ex.UnknownSettingKeyError)


# boolean_settings()

@pytest.mark.unittest
def test_boolean_settings_good():
    '''No errors when boolean settings are boolean
    '''
    conf = {'settinga': 'a', 'settingb': 'True', 'settingc': 'False'}
    bools = ['settingc', 'settingb']

    result = checkset.boolean_settings('testcomp', bools, conf)

    assert result == []


@pytest.mark.unittest
def test_boolean_settings_bad():
    '''Errors when boolean settings are not boolean
    '''
    conf = {'settinga': 'a', 'settingb': 'True', 'settingc': 'c'}
    bools = ['settinga', 'settingb']

    result = checkset.boolean_settings('testcomp', bools, conf)

    assert len(result) == 1
    for error in result:
        assert isinstance(error, ex.NotBooleanSettingError)


@pytest.mark.unittest
def test_boolean_settings_missing():
    '''No errors when the boolean setting is missing from the config
    '''
    conf = {}
    bools = ['settinga']

    result = checkset.boolean_settings('testcomp', bools, conf)

    assert result == []


# boolean_choice()

@pytest.mark.parametrize(
    ('name', 'config', 'error_count'), [
        ('settingname', {'settingname': 'yes-always'}, 0),
        ('settingname', {'settingname': 'choice-yes'}, 0),
        ('settingname', {'settingname': 'choice-no'}, 0),
        ('settingname', {'settingname': 'no-never'}, 0),
        ('settingname', {'settingname': 'unrecognized'}, 1),
        ('settingname', {}, 0)])
@pytest.mark.unittest
def test_boolean_choice(name, config, error_count):
    '''The right number of errors are returned
    '''
    result = checkset.boolean_choice('testcomp', [name], config)
    assert len(result) == error_count
