# Copyright (C) 2015, 2018, 2020, 2021 The Meme Factory, Inc.
# http://www.karlpinc.com/

# This file is part of PGWUI_Upload_Core.
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation, either version 3 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#

# Karl O. Pinc <kop@karlpinc.com>
#
# Bugs:
#  All data is presented to the db as a string, which could result
# in problems with type coercion.

import attr
import logging
import markupsafe
import psycopg2.errorcodes
from psycopg2 import ProgrammingError

from pgwui_core.core import (
    UploadNullFileInitialPost,
    DataLineProcessor,
    ParameterExecutor,
    TabularFileUploadHandler,
    UploadData,
    doublequote,
)
from pgwui_core.constants import (
    CSV,
    TAB,
)

from pgwui_upload_core import exceptions as upload_ex


log = logging.getLogger(__name__)


@attr.s
class UploadCoreInitialPost(UploadNullFileInitialPost):
    '''Get values from settings for when there's not user-supplied
    values in POST
    '''
    component = attr.ib(default='pgwui_upload')

    def set_component(self, component):
        self.component = component
        return self

    def boolean_choice(self, upload_settings, setting):
        val = upload_settings[setting]
        # Technically, we only need 'choice-yes' here because
        # otherwise the result is never displayed on the page.
        return (val == 'yes-always' or val == 'choice-yes')

    def find_upload_fmt(self, upload_settings):
        if upload_settings['file_format'] == 'csv':
            return CSV
        else:
            return TAB

    def build(self, settings={}):
        super().build(settings)
        upload_settings = settings['pgwui'][self.component]
        self.upload_fmt = self.find_upload_fmt(upload_settings)
        self.upload_null = self.boolean_choice(upload_settings, 'null')
        self.trim_upload = self.boolean_choice(upload_settings, 'trim')
        self.literal_col_headings = self.boolean_choice(
            upload_settings, 'literal_column_headings')
        return self


class SaveLine(DataLineProcessor, ParameterExecutor):
    def __init__(self, ue, uh, insert_stmt):
        '''
        ue             UploadEngine instance
        uh             UploadHandler instance
        insert_stmt    Statement used to insert into db.
                       (psycopg2 formatted for substituion)
        '''
        super(SaveLine, self).__init__(ue, uh)
        self.insert_stmt = insert_stmt

    def eat(self, udl):
        '''
        Upload a line of data into the db.

        udl  An UploadDataLine instance
        '''
        self.param_execute(self.insert_stmt, udl)


@attr.s
class BaseTableUploadHandler(TabularFileUploadHandler):
    '''
    Attributes:
      request       A pyramid request instance
      uf            A GCUploadForm instance
      session       A pyramid session instance
      ue
      cur
    '''
    ue = attr.ib(default=None)

    def get_data(self):
        '''
        Return an UploadData instance, with flags set as desired.
        '''
        uf = self.uf
        self.data = UploadData(uf['localfh'],
                               uf['upload_fmt'],
                               uf['upload_null'],
                               uf['null_rep'],
                               trim=uf['trim_upload'])

    def val_input(self):
        '''
        Validate input needed beyond that required to connect to the db.

        Returns:
          A list of PGWUIError instances
        '''
        errors = super().val_input()

        self.double_validator(errors)

        return errors

    def write(self, result, errors):
        '''Add double upload key into form.'''
        response = super().write(result, errors)
        self.write_double_key(response)
        return response

    def resolve_normalized_table(self, qualified_table):
        '''Return (schema, table) tuple of table name, or raise exception
        if not resolvable.
        '''
        try:
            self.cur.execute(
                ('SELECT nspname, relname'
                 '  FROM pg_class'
                 '       JOIN pg_namespace'
                 '            ON (pg_namespace.oid = pg_class.relnamespace)'
                 '  WHERE pg_class.oid = %s::REGCLASS::OID'),
                (qualified_table,))
        except ProgrammingError as err:
            pgcode = err.pgcode
            if pgcode == psycopg2.errorcodes.INVALID_SCHEMA_NAME:
                raise upload_ex.MissingSchemaError(
                    'No such schema',
                    err.diag.message_primary,)
            elif pgcode == psycopg2.errorcodes.UNDEFINED_TABLE:
                raise upload_ex.MissingTableError(
                    'No such table or view',
                    err.diag.message_primary,
                    ('<p>Hint: Check spelling, database permissions, '
                     ' or try qualifying the'
                     ' table name with a schema name</p>'))
            elif pgcode == psycopg2.errorcodes.INSUFFICIENT_PRIVILEGE:
                raise upload_ex.InsufficientPrivilegeError(
                    'Your PostgreSQL login has insufficient privileges',
                    err.diag.message_primary)
            else:
                raise
        return self.cur.fetchone()

    def resolve_table(self, qualified_table):
        '''Return (schema, table) tuple of table name or raise exception
        if character case is wrong
        '''
        (schema, table) = self.resolve_normalized_table(qualified_table)
        norm_qualified_table = qualified_table.lower()
        if schema:
            norm_schema = schema.lower()
            len_schema = len(schema)
            if (norm_qualified_table[0:len_schema] == norm_schema
                    and qualified_table[0:len_schema] != schema):
                raise upload_ex.MissingSchemaError(
                    'No such schema',
                    (f'The schema ({qualified_table[0:len_schema]}) does '
                     'not exist, but the same-but-for-character-case '
                     f'schema ({schema}) does'))
        norm_table = table.lower()
        len_table = len(table)
        if (norm_qualified_table[- len_table:] == norm_table
                and qualified_table[- len_table:] != table):
            raise upload_ex.MissingTableError(
                'No such table or view',
                (f'The table ({qualified_table[- len_table:]}) does '
                 'not exist, but the same-but-for-character-case '
                 f'table ({table}) does'))
        return (schema, table)

    def good_table(self, schema, table):
        '''Is the supplied table or view insertable?
        '''
        sql = ('SELECT 1 FROM information_schema.tables'
               '  WHERE tables.table_name = %s'
               '        AND tables.table_schema = %s'
               "        AND (tables.is_insertable_into = 'YES'"
               # Unfortunatly, as of 9.2, the information_schema
               # tables.is_insertable_into does not reflect whether
               # there's an insert trigger on the table.
               "             OR tables.table_type = 'VIEW')")
        self.cur.execute(sql, (table, schema))
        return self.cur.fetchone() is not None

    def quote_columns(self, settings):
        '''Return boolean -- whether to take column names literally
        '''
        quoter_setting = settings.get('literal_column_headings')
        return (quoter_setting == 'yes-always'
                or quoter_setting == 'choice-yes')

    def validate_table(self, qualified_table):
        '''Return schema and table names, or raise an exception
        if the relation is not writable
        '''
        schema, table = self.resolve_table(qualified_table)
        if not self.good_table(schema, table):
            raise upload_ex.CannotInsertError(
                'Cannot insert into supplied table or view',
                ('({0}) is either is a view'
                 ' that cannot be inserted into'
                 ' or you do not have the necessary'
                 ' permissions to the table or view').format(
                    markupsafe.escape(qualified_table)))
        return (schema, table)

    def report_bad_cols(self, qualified_table, bad_cols, quotecols):
        if quotecols:
            detail = ('<p>The following columns are not in the ({0})'
                      ' table, or the first line of your file is not'
                      ' in the expected format and your column names'
                      ' have merged (all the names appear in a single'
                      ' list item, below), or the supplied column names'
                      ' do not match'
                      " the character case of the table's columns,"
                      ' or you do not have permission to access'
                      ' the columns:</p><ul>')
        else:
            detail = ('<p>The following columns are not in the ({0})'
                      ' table, or the first line of your file is not'
                      ' in the expected format and your column names'
                      ' have merged (all the names appear in a single'
                      ' list item, below), '
                      ' or the table has column names containing'
                      ' upper case characters, or you do not have'
                      ' permission to access the columns:</p><ul>')
        detail = detail.format(markupsafe.escape(qualified_table))

        for bad_col in bad_cols:
            detail += '<li>{0}</li>'.format(markupsafe.escape(bad_col))
        detail += '</ul>'
        raise upload_ex.BadHeadersError(
            'Header line contains unknown column names',
            detail=detail)

    def get_column_quoter(self, quotecols):
        if quotecols:
            return doublequote
        else:
            def column_quoter(x):
                return x
            return column_quoter

    def quotetable(self, schema, table):
        if schema:
            return f'{doublequote(schema)}.{doublequote(table)}'
        return doublequote(table)

    def build_insert_stmt(
            self, data, qualified_table, quotecols, column_quoter):
        schema, table = self.validate_table(qualified_table)

        column_sql = ('SELECT 1 FROM information_schema.columns'
                      '  WHERE columns.table_name = %s'
                      '        AND columns.table_schema = %s')
        if quotecols:
            column_sql += '    AND columns.column_name = %s'
        else:
            column_sql += '    AND columns.column_name = lower(%s::name)'

        insert_stmt = f'INSERT INTO {self.quotetable(schema, table)} ('
        value_string = ''
        col_sep = ''
        bad_cols = []
        for col_name in data.headers.tuples:
            # Check that colum name exists
            self.cur.execute(column_sql, (table, schema, col_name))
            if self.cur.fetchone() is None:
                bad_cols.append(col_name)
            else:
                # Add column to sql statement
                insert_stmt += col_sep + column_quoter(col_name)
                value_string += col_sep + '%s'
                col_sep = ', '

        if bad_cols:
            self.report_bad_cols(qualified_table, bad_cols, quotecols)

        return insert_stmt + ') VALUES({0})'.format(value_string)

    def factory(self, ue):
        '''Make a db loader function from an UploadEngine.

        Input:

        Side Effects:
        Yes, lots.
        '''
        raise NotImplementedError()


def set_upload_response(component, request, response):
    '''Add to response per the upload component's settings
    Adds: ask_about_literal_cols
          ask_about_trim
          upload_settings
    '''
    settings = request.registry.settings
    upload_settings = settings['pgwui'][component]

    response.setdefault('pgwui', dict())
    response['pgwui']['upload_settings'] = upload_settings
