# Copyright (C) 2020, 2021 The Meme Factory, Inc.  http://www.karlpinc.com/

# This file is part of PGWUI_Upload_Core.
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation, either version 3 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#

# Karl O. Pinc <kop@karlpinc.com>

import pytest

import pgwui_upload_core.check_settings as check_settings

from pgwui_common import checkset
from pgwui_develop import testing

# Activiate the PGWUI pytest plugins
pytest_plugins = ("pgwui",)


# Mocks

mock_unknown_settings = testing.make_mock_fixture(
    checkset, 'unknown_settings')

mock_require_settings = testing.make_mock_fixture(
    checkset, 'require_settings')

mock_boolean_settings = testing.make_mock_fixture(
    checkset, 'boolean_settings')

mock_boolean_choice = testing.make_mock_fixture(
    checkset, 'boolean_choice')


# validate_file_format()

@pytest.mark.parametrize(
    ('setting', 'error_cnt'), [
        ('csv', 0),
        ('tab', 0),
        ('invalid', 1)])
@pytest.mark.unittest
def test_validate_file_format(setting, error_cnt):
    '''Returns the expected number of errors
    '''
    errors = []
    settings = {'file_format': setting}
    check_settings.validate_file_format(None, errors, settings)

    assert len(errors) == error_cnt


# check_settings()

@pytest.mark.unittest
def test_check_settings(mock_unknown_settings,
                        mock_require_settings,
                        mock_boolean_settings,
                        mock_boolean_choice):
    '''The setting checking functions are called once, the check_settings()
    call returns all the errors from each mock.
    '''

    unknown_retval = ['unk err']
    require_retval = ['req err']
    boolean_retval = ['bool err']
    boolean_choice_retval = ['choice err']

    mock_unknown_settings.return_value = unknown_retval
    mock_require_settings.return_value = require_retval
    mock_boolean_settings.return_value = boolean_retval
    mock_boolean_choice.return_value = boolean_choice_retval

    result = check_settings.check_settings(None, [], [], [], [], {})

    mock_unknown_settings.assert_called_once
    mock_require_settings.assert_called_once
    mock_boolean_settings.assert_called_once
    mock_boolean_choice.assert_called_once

    assert result.sort() == (unknown_retval
                             + require_retval
                             + boolean_retval
                             + boolean_choice_retval).sort()
